/* Hijerarhija klasa kojom mozemo predstaviti matematicke funkcije jedne promenljive. Hijerarhija moze biti sledeca:
 * 
 * Funkcija ---> KonstantnaFunkcija
 *          ---> IdentitetFunkcija
 *          ---> BinarniOperator    ---> PlusFunkcija
 *                                  ---> MinusFunkcija
 *                                  ---> PutaFunkcija
 *                                  ---> PodeljenoFunkcija
 *          ---> ElementarnaFunkcija    ---> SuprotnaFunkcija
 *                                      ---> SinFunkcija
 *                                      ---> CosFunkcija
 *                                      ---> TanFunkcija
 *                                      ---> CotFunkcija
 *                                      ---> LogFunkcija
 *                                      ---> ExpFunkcija
 *                                      ---> PowFunkcija
 *                                      ---> SqrtFunkcija
 * BITNO:
 * Prilikom razvoja hijerarhije klasa za predstavljanje Funkcija razmisljajte unapred na koji nacin
 * cete je koristiti prilikom izgradnje sintaksnog stabla. Cilj je da razvijenu hijerarhiju klasa
 * mozemo u nepromenjenom obliku da koristimo i u interaktivnom parseru i u parseru koji izgradjuje
 * apstraktno stablo sintakse. 
 */

#ifndef FUNKCIJE_H
#define FUNKCIJE_H

#include <iostream>
#include <cmath>

/* bazna klasa hijerarhije */
class Funkcija {
public:
    /* virtuelni desktruktor zbog pravilnog nadovezivanja destruktora prilikom unistavanja objekata */
    virtual ~Funkcija() {}

    /* metod stampa string reprezentaciju funkcije na strim s*/
    virtual void Print(std::ostream& s) const = 0;
    /* metod pravi identicnu kopiju objekta */
    virtual Funkcija* Clone() const = 0;
    /* metod racuna vrednost funkcije u tacki x */
    virtual double Value(double x) const = 0;
    /* metod odreduje izvod funkcije */
    virtual Funkcija* Diff() const = 0;
    /* metod kreira novu funkciju koja je kompozicija postojece i funkcije f */
    virtual Funkcija* Compose(Funkcija* f) const = 0;
};

/* preoptereceni operator stampanja */
std::ostream& operator <<(std::ostream& s, const Funkcija& f);

/* klasa predstavlja konstantnu funkciju */
class KonstantnaFunkcija : public Funkcija {

private:
    /* vrednost funkcije */
    double _value;
public:
    KonstantnaFunkcija(double val);
    KonstantnaFunkcija(const KonstantnaFunkcija& kf);

    virtual void Print(std::ostream& s) const;
    virtual Funkcija* Clone() const;
    virtual double Value(double x) const;
    virtual Funkcija* Diff() const;
    virtual Funkcija* Compose(Funkcija* f) const ;
};

/* klasa predstavlja identicku funkciju, tj. f(x) = x. */
class IdentitetFunkcija : public Funkcija {

public:

    virtual void Print(std::ostream& s) const;
    virtual Funkcija* Clone() const;
    virtual double Value(double x) const;
    virtual Funkcija* Diff() const;
    virtual Funkcija* Compose(Funkcija* f) const ;
};

/* bazna klasa za sve binarne operatore. */
class BinarniOperator : public Funkcija {

protected:
    /* simbol, levi i desni operand su atributi*/
    std::string _symbol;
    /* operande pamtimo kao pokazivace, jer nam treba polimorfno ponasanje */
    Funkcija* _left;
    Funkcija* _right;
public:
    BinarniOperator(const std::string& s, Funkcija* l, Funkcija* r);
    ~BinarniOperator();

    virtual void Print(std::ostream& s) const;
    virtual Funkcija* Clone() const = 0;
    virtual double Value(double x) const = 0;
    virtual Funkcija* Diff() const = 0;
    virtual Funkcija* Compose(Funkcija* f) const = 0;
};

/* specijalicije binarnog operatora */

class PlusFunkcija : public BinarniOperator {

public:
    /* klase su konkretne, pa imaju konstruktor i konstruktor kopije neophodan za kloniranje */
    PlusFunkcija(Funkcija* l, Funkcija* d);
    PlusFunkcija(const PlusFunkcija& pf);

    virtual Funkcija* Clone() const;
    virtual double Value(double x) const;
    virtual Funkcija* Diff() const;
    virtual Funkcija* Compose(Funkcija* f) const;
};

class MinusFunkcija : public BinarniOperator {

public:
    MinusFunkcija(Funkcija* l, Funkcija* d);
    MinusFunkcija(const MinusFunkcija& pf);

    virtual Funkcija* Clone() const;
    virtual double Value(double x) const;
    virtual Funkcija* Diff() const;
    virtual Funkcija* Compose(Funkcija* f) const;
};

class PutaFunkcija : public BinarniOperator {

public:
    PutaFunkcija(Funkcija* l, Funkcija* d);
    PutaFunkcija(const PutaFunkcija& pf);

    virtual Funkcija* Clone() const;
    virtual double Value(double x) const;
    virtual Funkcija* Diff() const;
    virtual Funkcija* Compose(Funkcija* f) const;
};

class PodeljenoFunkcija : public BinarniOperator {

public:
    PodeljenoFunkcija(Funkcija* l, Funkcija* d);
    PodeljenoFunkcija(const PodeljenoFunkcija& pf);

    virtual Funkcija* Clone() const;
    virtual double Value(double x) const;
    virtual Funkcija* Diff() const;
    virtual Funkcija* Compose(Funkcija* f) const;
};

/* bazna klasa za sve elementarne funkcije */
class ElementarnaFunkcija : public Funkcija {

protected:
    /* naziv funkcije */
    std::string _name;
    /* izraz na koji treba primeniti funkciju */
    Funkcija* _expr;
public:
    ElementarnaFunkcija(const std::string& s, Funkcija* f);
    ~ElementarnaFunkcija();

    virtual void Print(std::ostream& s) const;
    virtual Funkcija* Clone() const = 0;
    virtual double Value(double x) const = 0;
    virtual Funkcija* Diff() const = 0;
    virtual Funkcija* Compose(Funkcija* f) const = 0;
};

/* specijalicije elementarne funkcije */

class SuprotnaFunkcija : public ElementarnaFunkcija {
public:
    /* klase su konkretne, pa imaju konstruktor i konstruktor kopije neophodan za kloniranje */
    SuprotnaFunkcija(Funkcija* expr);
    SuprotnaFunkcija(const SuprotnaFunkcija& sf);

    virtual Funkcija* Clone() const;
    virtual double Value(double x) const;
    virtual Funkcija* Diff() const;
    virtual Funkcija* Compose(Funkcija* f) const;
};

class SinFunkcija : public ElementarnaFunkcija {
public:
    SinFunkcija(Funkcija* expr);
    SinFunkcija(const SinFunkcija& sf);

    virtual Funkcija* Clone() const;
    virtual double Value(double x) const;
    virtual Funkcija* Diff() const;
    virtual Funkcija* Compose(Funkcija* f) const;
};

class CosFunkcija : public ElementarnaFunkcija {
public:
    CosFunkcija(Funkcija* expr);
    CosFunkcija(const CosFunkcija& sf);

    virtual Funkcija* Clone() const;
    virtual double Value(double x) const;
    virtual Funkcija* Diff() const;
    virtual Funkcija* Compose(Funkcija* f) const;
};

class TanFunkcija : public ElementarnaFunkcija {
public:
    TanFunkcija(Funkcija* expr);
    TanFunkcija(const TanFunkcija& sf);

    virtual Funkcija* Clone() const;
    virtual double Value(double x) const;
    virtual Funkcija* Diff() const;
    virtual Funkcija* Compose(Funkcija* f) const;
};

class CotFunkcija : public ElementarnaFunkcija {
public:
    CotFunkcija(Funkcija* expr);
    CotFunkcija(const CotFunkcija& sf);

    virtual Funkcija* Clone() const;
    virtual double Value(double x) const;
    virtual Funkcija* Diff() const;
    virtual Funkcija* Compose(Funkcija* f) const;
};

class ExpFunkcija : public ElementarnaFunkcija {
public:
    ExpFunkcija(Funkcija* expr);
    ExpFunkcija(const ExpFunkcija& sf);

    virtual Funkcija* Clone() const;
    virtual double Value(double x) const;
    virtual Funkcija* Diff() const;
    virtual Funkcija* Compose(Funkcija* f) const;
};

class LogFunkcija : public ElementarnaFunkcija {
public:
    LogFunkcija(Funkcija* expr);
    LogFunkcija(const LogFunkcija& sf);

    virtual Funkcija* Clone() const;
    virtual double Value(double x) const;
    virtual Funkcija* Diff() const;
    virtual Funkcija* Compose(Funkcija* f) const;
};

class PowFunkcija : public ElementarnaFunkcija {
private:
    double _stepen;
public:
    PowFunkcija(Funkcija* expr, double step);
    PowFunkcija(const PowFunkcija& sf);

    virtual Funkcija* Clone() const;
    virtual double Value(double x) const;
    virtual Funkcija* Diff() const;
    virtual Funkcija* Compose(Funkcija* f) const;
};

class SqrtFunkcija : public ElementarnaFunkcija {
public:
    SqrtFunkcija(Funkcija* expr);
    SqrtFunkcija(const SqrtFunkcija& sf);

    virtual Funkcija* Clone() const;
    virtual double Value(double x) const;
    virtual Funkcija* Diff() const;
    virtual Funkcija* Compose(Funkcija* f) const;
};

#endif
